﻿/* 
# Uses packages:
#     Newtonsoft.Json;
#
# Usage:
#     Download API key from https://app.aplos.com/aws/settings/api/configure
#     This should result in a file with the name (aplos_id.key).
#     Put that file in the same directory as this C# code file.
#     Update the api_id value with your api_key value.
#
#
*/
using Newtonsoft.Json;
using System.Net;
using System.Net.Http.Headers;
using System.Security.Cryptography;

namespace AplosApiExample
{


  public class JsonRoot
  {
    public string? version { get; set; }
    public string? message { get; set; }
    public int? status { get; set; }
    public Exception? exception { get; set; }
    public jsonData? data { get; set; }
  }

  public class jsonException
  {
    public string? message { get; set; }
    public int code { get; set; }
  }

  public class jsonData
  {
    public DateTime expires { get; set; }
    public string? token { get; set; }
    public Account[]? accounts { get; set; }
    public Transaction[]? transactions { get; set; }
    public Contact[]? contacts { get; set; }
  }

  public class Account
  {
    [JsonProperty("account_number")]
    public string? AccountNumber { get; set; }
    public string? Name { get; set; }
    public string? Category { get; set; }
    [JsonProperty("account_group")]
    public AccountGroup? AccountGroup { get; set; }
    [JsonProperty("is_enabled")]
    public bool IsEnabled { get; set; }
    public string? Type { get; set; }
    public string? Activity { get; set; }
  }

  public class AccountGroup
  {
    public int Id { get; set; }
    public string? Name { get; set; }
    public int Seq { get; set; }
  }

  public class Transaction
  {
    public int Id { get; set; }
    public string? Note { get; set; }
    public string? Date { get; set; }
    public Contact? Contact { get; set; }
    public string? Created { get; set; }
    public string? Modified { get; set; }
    public decimal Amount { get; set; }
    [JsonProperty("in_closed_period")]
    public bool InClosedPeriod { get; set; }
  }

  public class Contact
  {
    public int Id { get; set; }
    [JsonProperty("company_name")]
    public string? CompanyName { get; set; }
    public string? Type { get; set; }
    public string? Email { get; set; }
    [JsonProperty("is_accounting")]
    public bool IsAccounting { get; set; }
    [JsonProperty("is_dm")]
    public bool IsDm { get; set; }
  }

  public class AplosApiExample
  {

    public static string ReadFile(string apikey_filename)
    {
      string? api_user_key = null;
      try
      {   // Open the text file using a stream reader.
        using StreamReader sr = new(apikey_filename);
        api_user_key = sr.ReadToEnd();
      }
      catch (Exception e)
      {
        Console.WriteLine("The file could not be read:");
        Console.WriteLine(e.Message);
      }

      return api_user_key;
    }

    public static byte[] GetBytes(string str)
    {
      byte[] bytes = new byte[str.Length * sizeof(char)];
      System.Buffer.BlockCopy(str.ToCharArray(), 0, bytes, 0, bytes.Length);
      return bytes;
    }

    public static string GetString(byte[] bytes)
    {
      char[] chars = new char[bytes.Length / sizeof(char)];
      System.Buffer.BlockCopy(bytes, 0, chars, 0, bytes.Length);
      return new string(chars);
    }

    public static string EncryptData(string SecretMessage, RSACryptoServiceProvider rsaObj)
    {

      byte[] encryptedBytes;
      byte[] messageBytes = GetBytes(SecretMessage);

      try
      {
        encryptedBytes = rsaObj.Encrypt(messageBytes, false);
      }
      catch
      {
        throw new CryptographicException("Unable to encrypt data.");
      }

      //    Check to make sure we decrpyted the string 
      if (encryptedBytes.Length == 0)
        return String.Empty;
      else
      {
        string encryptedString = Convert.ToBase64String(encryptedBytes);
        Console.WriteLine("Read this batman:");
        Console.WriteLine(encryptedString);
        return encryptedString;
      }
    }

    public static string DecryptEncryptedData(string Base64EncryptedData, RSACryptoServiceProvider rsaObj)
    {
      byte[] decryptedBytes;
      try
      {
        decryptedBytes = rsaObj.Decrypt(Convert.FromBase64String(Base64EncryptedData), false);
      }
      catch
      {
        throw new CryptographicException("Unable to decrypt data.");
      }

      //    Check to make sure we decrpyted the string 
      if (decryptedBytes.Length == 0)
        return String.Empty;
      else
      {
        string decryptedString = System.Text.Encoding.ASCII.GetString(decryptedBytes);
        Console.WriteLine("Decrypted: " + decryptedString);
        return decryptedString;
      }
    }


    private static async Task<T?> DownloadSerializedJsonData<T>(string url, string path, string api_id) where T : new()
    {
      using var client = new HttpClient();
      var json_data = string.Empty;
      // attempt to download JSON data as a string
      try
      {
        ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;
        HttpResponseMessage response = await client.GetAsync(url + path + api_id);
        response.EnsureSuccessStatusCode();
        json_data = await response.Content.ReadAsStringAsync();
        // Console.WriteLine(json_data);
      }
      catch (Exception) { }
      // if string with JSON data is not empty, deserialize it to class and return its instance 
      return !string.IsNullOrEmpty(json_data) ? JsonConvert.DeserializeObject<T>(json_data) : new T();
    }

    private static async Task<T?> DownloadSerializedJsonData<T>(string url, string path, string api_id, string token) where T : new()
    {
      using var client = new HttpClient();
      var json_data = string.Empty;
      // attempt to download JSON data as a string
      try
      {
        ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

        client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", token);

        HttpResponseMessage response = await client.GetAsync(url + path + api_id);

        response.EnsureSuccessStatusCode();

        json_data = await response.Content.ReadAsStringAsync();
      }
      catch (Exception ex)
      {
        Console.WriteLine(ex.ToString());
      }
      // if string with JSON data is not empty, deserialize it to class and return its instance 
      return !string.IsNullOrEmpty(json_data) ? JsonConvert.DeserializeObject<T>(json_data) : new T();
    }

    public static async Task Main()
    {
      string api_base_url = "https://app.aplos.com/hermes/api/v1/";
      string api_id = "[YOUR API KEY HERE]";

      RSACryptoServiceProvider rsa = new();
      string apikey_filename = api_id + ".key";
      StreamReader sr = new(apikey_filename);
      String api_user_key = "-----BEGIN PRIVATE KEY-----" + System.Environment.NewLine;
      api_user_key += sr.ReadToEnd().Trim();
      api_user_key += System.Environment.NewLine + "-----END PRIVATE KEY-----";
      Console.Write(api_user_key);
      rsa.ImportFromPem(api_user_key);

      // Test encryption and decryption here.
      // string etoken = EncryptData("This is the road to Iceland", rsa);
      // string dtoken = DecryptEncryptedData(etoken, rsa);
      // Console.WriteLine("dtoken: " + dtoken);

      //Authenticate
      JsonRoot auth = await DownloadSerializedJsonData<JsonRoot>(api_base_url, "auth/", api_id);
      jsonData d = auth.data;
      Console.WriteLine("Web client status:" + auth.status);
      Console.WriteLine("Token Expires:" + d.expires);
      string authtoken = DecryptEncryptedData(d.token, rsa);

      // //Fetch contacts
      JsonRoot contactsResponse = await DownloadSerializedJsonData<JsonRoot>(api_base_url, "contacts/", api_id, authtoken);
      Console.WriteLine("Web client status:" + contactsResponse?.status);
      var contactsArray = contactsResponse?.data?.contacts;
      if (contactsArray != null)
      {
        // Log the contacts to the console.
        Console.WriteLine("\n-----------Contacts-----------");
        foreach (var contact in contactsArray)
        {
          Console.WriteLine($"Contact ID: {contact?.Id}");
          Console.WriteLine($"Company Name: {contact?.CompanyName}");
          Console.WriteLine($"Type: {contact?.Type}");
          Console.WriteLine($"Email: {contact?.Email}");
          Console.WriteLine($"Is Accounting: {contact?.IsAccounting}");
          Console.WriteLine($"Is DM: {contact?.IsDm}");
          Console.WriteLine();

        }
      }

      //Fetch transactions
      JsonRoot transactionsResponse = await DownloadSerializedJsonData<JsonRoot>(api_base_url, "transactions/", api_id, authtoken);
      Console.WriteLine("Web client status:" + transactionsResponse?.status);
      var transactionsArray = transactionsResponse?.data?.transactions;
      if (transactionsArray != null)
      {
        // Log the transactions to the console.
        Console.WriteLine("\n-----------Transactions-----------");
        foreach (var transaction in transactionsArray)
        {
          Console.WriteLine($"Transaction ID: {transaction.Id}");
          Console.WriteLine($"Note: {transaction.Note}");
          Console.WriteLine($"Date: {transaction.Date}");
          Console.WriteLine($"Contact ID: {transaction.Contact?.Id}");
          Console.WriteLine($"Company Name: {transaction.Contact?.CompanyName}");
          Console.WriteLine($"Type: {transaction.Contact?.Type}");
          Console.WriteLine($"Email: {transaction.Contact?.Email}");
          Console.WriteLine($"Is Accounting: {transaction.Contact?.IsAccounting}");
          Console.WriteLine($"Is DM: {transaction.Contact?.IsDm}");
          Console.WriteLine($"Created: {transaction.Created}");
          Console.WriteLine($"Modified: {transaction.Modified}");
          Console.WriteLine($"Amount: {transaction.Amount}");
          Console.WriteLine($"In Closed Period: {transaction.InClosedPeriod}");
          Console.WriteLine();
        }
      }
      //     //Fetch accounts
      JsonRoot accountsResponse = await DownloadSerializedJsonData<JsonRoot>(api_base_url, "accounts/", api_id, authtoken);
      Console.WriteLine("Web client status:" + accountsResponse?.status);

      var accountsArray = accountsResponse?.data?.accounts;

      if (accountsArray != null)
      {
        // Log the accounts to the console.
        Console.WriteLine("\n-----------Accounts-----------");
        foreach (var account in accountsArray)
        {
          Console.WriteLine($"Account Number: {account.AccountNumber}");
          Console.WriteLine($"Name: {account.Name}");
          Console.WriteLine($"Category: {account.Category}");
          Console.WriteLine($"Account Group ID: {account.AccountGroup?.Id}");
          Console.WriteLine($"Account Group Name: {account.AccountGroup?.Name}");
          Console.WriteLine($"Account Group Seq: {account.AccountGroup?.Seq}");
          Console.WriteLine($"Is Enabled: {account.IsEnabled}");
          Console.WriteLine($"Type: {account.Type}");
          Console.WriteLine($"Activity: {account.Activity}");
          Console.WriteLine();
        }
      }
    }

  }
}
