#!/usr/bin/env python

##############################
# ['certifi==2021.10.8', 'chardet==4.0.0', 'idna==2.10', 'pip==20.3.4', 'pyasn1==0.2.1', 'python==2.7.18', 'requests==2.27.1', 'rsa==3.2', 'setuptools==41.2.0', 'urllib3==1.26.16', 'wsgiref==0.1.2']
#
# Uses packages:
#     requests (http://docs.python-requests.org/en/latest/)
#     rsa (http://stuvel.eu/files/python-rsa-doc/)
#
# Requires application:
#     openssl (for converting keys from PKCS8 to PKCS1)
#
# Usage:
#     Download API key from https://app.aplos.com/aws/settings/api/configure
#     This should result in a file with the name (aplos_id.key).
#     Put that file in the same directory as this python script, adding the extension:
#     .download, so your new file name will be (aplos_id.key.download).
#     Update the api_id value with your api_key value.
#
#
##############################

import base64
import json
import os
import textwrap
import subprocess
import sys
import select

try:
    from requests import requests
except:
    import requests

try:
    from rsa import rsa
except:
    import rsa

import pkg_resources

installed_packages = pkg_resources.working_set
installed_packages_list = sorted(["%s==%s" % (i.key, i.version)
    for i in installed_packages])
print(installed_packages_list)

# Lets do some setup here...
# production
api_base_url = 'https://app.aplos.com/hermes/api/v1/'
api_id = '[YOUR API KEY HERE]'

def convert_pkcs8_to_pkcs1(api_id):
    # Lets convert the raw downloaded, unformatted key to
    # a formatted pkcs8 file.
    with open(api_id + '.key.download', mode='rb') as pkcs8file:
        api_user_pkcs8file = '-----BEGIN PRIVATE KEY-----\n' + \
            textwrap.fill(pkcs8file.read().decode('utf-8'), 64) + \
            '\n-----END PRIVATE KEY-----\n\r'
    with open(api_id + '.pkcs8', mode='w') as pkcs1keyfile:
        pkcs1keyfile.write(api_user_pkcs8file)

    # Now for some OpenSSL magic, convert pkcs8 to pkcs1 so we can actually
    # use it. Depending on your OpenSSL this may change.
    
    p = subprocess.Popen(['openssl', 'version'], stdout=subprocess.PIPE)
    t = p.stdout.read()
    print(t)
    # openssl version is greater than 3.0.0 for this code
    openssl_cvrt_cmd = 'openssl pkey -in {}.pkcs8 -traditional -out  {}.key'.format(api_id, api_id)

    p = subprocess.Popen(openssl_cvrt_cmd, shell=True, stderr=subprocess.PIPE)
    while p.poll() is None: # while the process is running
        reads, _, _ = select.select([p.stderr], [], [], 0.1) # check if there's anything to read
        if p.stderr in reads:
            out = p.stderr.read(1)
            if out != b'':
                sys.stdout.write(out.decode('utf-8'))
                sys.stdout.flush()
    with open(api_id + '.key', mode='rb') as pkeyfile:
        api_user_pemkey = pkeyfile.read()
    return(api_user_pemkey)


def api_error_handling(status_code):
    # Some Basic Error Handling:
    # Check for HTTP codes other than 200
    if status_code != 200:
        if status_code == 401:
            print('Status:', status_code, 'Something is wrong with the auth code. Exiting')
            exit()
        elif status_code == 403:
            print('Status:', status_code, 'Forbidden. Exiting')
            exit()
        elif status_code == 405:
            print('Status:', status_code, 'Method not allowed. Exiting')
            exit()
        elif status_code == 422:
            print('Status:', status_code, 'Unprocessable Entity. Exiting')
            exit()
        print('Status:', status_code, 'Problem with the request. Exiting.')
        exit()
    else:
        print('Status:', status_code, ': The API let me in!')
    return()


def api_auth(api_base_url, api_id, api_user_key):
    # This should return an authorized token so we can do other, more exciting things
    # Lets show what we're doing.
    print('geting URL: {}auth/{}'.format(api_base_url, api_id))

    # Actual request goes here.
    r = requests.get('{}auth/{}'.format(api_base_url, api_id))
    data = r.json()
    api_error_handling(r.status_code)

    api_token_encrypted = data['data']['token']
    api_token_encrypted_expires = data['data']['expires']
    print('The API Token expires: {}'.format(api_token_encrypted_expires))

    api_bearer_token = rsa.decrypt(
        base64.decodebytes(api_token_encrypted.encode()), api_user_key)
    return(api_bearer_token.decode('utf-8'))


def api_contacts_get(api_base_url, api_id, api_access_token):
    # This should print(a contact from Aplos.)
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print('geting URL: {}contacts'.format(api_base_url))
    print('With headers: {}'.format(headers))

    # Actual request goes here.
    r = requests.get('{}contacts'.format(api_base_url), headers=headers)
    print(r)
    api_error_handling(r.status_code)
    response = r.json()
    print('JSON response: {}'.format(response))
    return (response)


def api_accounts_get(api_base_url, api_id, api_access_token):
    # This should print(a contact from Aplos.)
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print('geting URL: {}accounts'.format(api_base_url))
    print('With headers: {}'.format(headers))

    # Actual request goes here.
    r = requests.get('{}accounts'.format(api_base_url), headers=headers)
    api_error_handling(r.status_code)
    response = r.json()
    print('JSON response: {}'.format(response))
    return (response)


def api_transactions_get(api_base_url, api_id, api_access_token):
    # This should print(a contact from Aplos.)
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print('geting URL: {}transactions'.format(api_base_url))
    print('With headers: {}'.format(headers))

    # Actual request goes here.
    r = requests.get('{}transactions'.format(api_base_url), headers=headers)
    api_error_handling(r.status_code)
    response = r.json()

    print('JSON response: {}'.format(response))
    return (response)


def api_transactions_post(api_base_url, api_id, api_access_token):
    # This should print(a contact from Aplos.)
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print('Posting URL: {}transactions'.format(api_base_url))
    print('With headers: {}'.format(headers))

    # Actual request goes here.
    # Note, these values must match extant data so you need fund: 647 and
    # accounts, 1000 and 4002. If these aren't there, you will get a 422.
    payload = '{"note": "Insert a transaction", ' \
                '"date": "2015-06-01", ' \
                '"contact": { ' \
                    '"companyname": "Aplos", ' \
                    '"type": "company" }, ' \
                '"lines": [ { ' \
                    '"amount": 123.45, ' \
                    '"account": { ' \
                        '"account_number": 1010 }, ' \
                        '"fund": { ' \
                            '"id": 647 } }, ' \
                    '{ ' \
                    '"amount": -123.45, ' \
                    '"account": { ' \
                        '"account_number": 4002 }, ' \
                        '"fund": { ' \
                            '"id": 647 ' \
                '} } ] }'
    print(payload)
    r = requests.post(
        '{}transactions'.format(api_base_url), headers=headers, data=payload)
    api_error_handling(r.status_code)
    response = r.json()
    print('JSON response: {}'.format(response))
    return (response)

# Setup RSA and import the PCKS1 version of the keyfile.
try:
    if os.stat(api_id + '.key').st_size > 0:
        print('Key file exists\n')
        with open(api_id + '.key', mode='rb') as pkeyfile:
            api_user_pemkey = pkeyfile.read()
    else:
        print('Key file empty\n')
        api_user_pemkey = convert_pkcs8_to_pkcs1(api_id)
except OSError:
    print('No file\n')
    api_user_pemkey = convert_pkcs8_to_pkcs1(api_id)
api_user_key = rsa.PrivateKey.load_pkcs1(api_user_pemkey)

# Now do some API manipulation
api_access_token = api_auth(api_base_url, api_id, api_user_key)
contacts = api_contacts_get(api_base_url, api_id, api_access_token)
# transactions = api_transactions_get(api_base_url, api_id, api_access_token)
# accounts = api_accounts_get(api_base_url, api_id, api_access_token)
# transaction_post = api_transactions_post(api_base_url, api_id, api_access_token)
