#!/usr/bin/env python

# Uses packages:
#     requests (http://docs.python-requests.org/en/latest/)
#     rsa (http://stuvel.eu/files/python-rsa-doc/)
#
# Requires application:
#     openssl (for converting keys from PKCS8 to PKCS1)
#
# Usage:
#     Download API key from https://app.aplos.com/aws/settings/api/configure
#     This should result in a file with the name (aplos_id.key).
#     Put that file in the same directory as this python script, adding the extension:
#     .download, so your new file name will be (aplos_id.key.download).
#     Update the api_id value with your api_key value.
#
#
##############################

import base64
import json
import os
import textwrap
import subprocess
import sys
import requests
import rsa

import pkg_resources

# Lets do some setup here...
# production
api_base_url = 'https://app.aplos.com/hermes/api/v1/'
api_id = '[YOUR API KEY HERE]'

def convert_pkcs8_to_pkcs1(api_id):
    # Lets convert the raw downloaded, unformatted key to
    # a formatted pkcs8 file.
    with open(api_id + '.key.download', mode='rb') as pkcs8file:
        api_user_pkcs8file = '-----BEGIN PRIVATE KEY-----\n' + \
            textwrap.fill(pkcs8file.read(), 64) + \
            '\n-----END PRIVATE KEY-----\n\r'
    with open(api_id + '.pkcs8', mode='w') as pkcs1keyfile:
        pkcs1keyfile.write(api_user_pkcs8file)

    # Now for some OpenSSL magic, convert pkcs8 to pkcs1 so we can actually
    # use it. Depending on your OpenSSL this may change.
    #
    # OpenSSL > 3.0.0 is required for this to work.
    p = subprocess.Popen(['openssl', 'version'], stdout=subprocess.PIPE)
    t = p.stdout.read()
    print t
 
    openssl_cvrt_cmd = 'openssl pkey -in {}.pkcs8 -traditional -out  {}.key'.format(
        api_id, api_id)

    print openssl_cvrt_cmd
    p = subprocess.Popen(openssl_cvrt_cmd, shell=True, stderr=subprocess.PIPE)
    while True:
        out = p.stderr.read(1)
        if out == '' and p.poll() != None:
            break
        if out != '':
            sys.stdout.write(out)
            sys.stdout.flush()

    with open(api_id + '.key', mode='rb') as pkeyfile:
        api_user_pemkey = pkeyfile.read()
    return(api_user_pemkey)


def api_error_handling(status_code):
    # Some Basic Error Handling:
    # Check for HTTP codes other than 200
    if status_code != 200:
        if status_code == 401:
            print 'Status:', status_code, 'Something is wrong with the auth code. Exiting'
            exit()
        elif status_code == 403:
            print 'Status:', status_code, 'Forbidden. Exiting'
            exit()
        elif status_code == 405:
            print 'Status:', status_code, 'Method not allowed. Exiting'
            exit()
        elif status_code == 422:
            print 'Status:', status_code, 'Unprocessable Entity. Exiting'
            exit()
        print 'Status:', status_code, 'Problem with the request. Exiting.'
        exit()
    else:
        print 'Status:', status_code, ': The API let me in!'
    return()


def api_auth(api_base_url, api_id, api_user_key):
    # This should return an authorized token so we can do other, more exciting things
    # Lets show what we're doing.
    print 'geting URL: {}auth/{}'.format(api_base_url, api_id)

    # Actual request goes here.
    r = requests.get('{}auth/{}'.format(api_base_url, api_id))
    data = r.json()
    api_error_handling(r.status_code)

    api_token_encrypted = data['data']['token']
    api_token_encrypted_expires = data['data']['expires']
    print 'The API Token expires: {}'.format(api_token_encrypted_expires)

    api_bearer_token = rsa.decrypt(
        base64.decodestring(api_token_encrypted), api_user_key)
    return(api_bearer_token)


def api_contacts_get(api_base_url, api_id, api_access_token):
    # This should print a contact from Aplos.
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print 'geting URL: {}contacts'.format(api_base_url)
    print 'With headers: {}'.format(headers)

    # Actual request goes here.
    r = requests.get('{}contacts'.format(api_base_url), headers=headers)
    api_error_handling(r.status_code)
    response = r.json()
    print 'JSON response: {}'.format(response)
    return (response)


def api_accounts_get(api_base_url, api_id, api_access_token):
    # This should print a contact from Aplos.
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print 'geting URL: {}accounts'.format(api_base_url)
    print 'With headers: {}'.format(headers)

    # Actual request goes here.
    r = requests.get('{}accounts'.format(api_base_url), headers=headers)
    api_error_handling(r.status_code)
    response = r.json()
    print 'JSON response: {}'.format(response)
    return (response)


def api_transactions_get(api_base_url, api_id, api_access_token):
    # This should print a contact from Aplos.
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print 'geting URL: {}transactions'.format(api_base_url)
    print 'With headers: {}'.format(headers)

    # Actual request goes here.
    r = requests.get('{}transactions'.format(api_base_url), headers=headers)
    api_error_handling(r.status_code)
    response = r.json()

    print 'JSON response: {}'.format(response)
    return (response)


def api_transactions_post(api_base_url, api_id, api_access_token):
    # This should print a contact from Aplos.
    # Lets show what we're doing.
    headers = {'Authorization': 'Bearer: {}'.format(api_access_token)}
    print 'Posting URL: {}transactions'.format(api_base_url)
    print 'With headers: {}'.format(headers)

    # Actual request goes here.
    # Note, these values must match extant data so you need fund: 647 and
    # accounts, 1000 and 4002. If these aren't there, you will get a 422.
    payload = '{"note": "Insert a transaction", ' \
                '"date": "2015-06-01", ' \
                '"contact": { ' \
                    '"companyname": "Aplos", ' \
                    '"type": "company" }, ' \
                '"lines": [ { ' \
                    '"amount": 123.45, ' \
                    '"account": { ' \
                        '"account_number": 1010 }, ' \
                        '"fund": { ' \
                            '"id": 647 } }, ' \
                    '{ ' \
                    '"amount": -123.45, ' \
                    '"account": { ' \
                        '"account_number": 4002 }, ' \
                        '"fund": { ' \
                            '"id": 647 ' \
                '} } ] }'
    print payload
    r = requests.post(
        '{}transactions'.format(api_base_url), headers=headers, data=payload)
    api_error_handling(r.status_code)
    response = r.json()
    print 'JSON response: {}'.format(response)
    return (response)

# Setup RSA and import the PCKS1 version of the keyfile.
try:
    if os.stat(api_id + '.key').st_size > 0:
        print 'Key file exists\n'
        with open(api_id + '.key', mode='rb') as pkeyfile:
            api_user_pemkey = pkeyfile.read()
    else:
        print 'Key file empty\n'
        api_user_pemkey = convert_pkcs8_to_pkcs1(api_id)
except OSError:
    print 'No file\n'
    api_user_pemkey = convert_pkcs8_to_pkcs1(api_id)
print api_user_pemkey
api_user_key = rsa.PrivateKey.load_pkcs1(api_user_pemkey)

# Now do some API manipulation
api_access_token = api_auth(api_base_url, api_id, api_user_key)
#contacts = api_contacts_get(api_base_url, api_id, api_access_token)
#transactions = api_transactions_get(api_base_url, api_id, api_access_token)
accounts = api_accounts_get(api_base_url, api_id, api_access_token)
#transaction_post = api_transactions_post(api_base_url, api_id, api_access_token)
